<?php defined('BASEPATH') OR exit('No direct script access allowed');

// -------------- Base Controller -------------- //
class Base_Controller extends CI_Controller
{
   protected $headers = FALSE;

   //router Data
   protected $ct_directory = '/';    // director of folder
   protected $ct_ctrl = 'home';	   // current controller
   protected $ct_action = 'index';   // controller function being called
   protected $ct_method = 'GET';	   // HTTP request method

   /**
    * Header Request String
    */
   protected $l_authorization = 'authorization';
   protected $u_authorization = 'Authorization';

   /**
    * HTTP STATUS CODE
    */
   const HTTP_OK = 200; // The request has succeeded
   const HTTP_BAD_REQUEST = 400; // The request cannot be fulfilled due to multiple errors
   const HTTP_UNAUTHORIZED = 401; // The user is unauthorized to access the requested resource
   const HTTP_PAYMENT_REQUIRED = 402;
   const HTTP_FORBIDDEN = 403; // The requested resource is unavailable at this present time
   const HTTP_NOT_FOUND = 404; // The requested resource could not be found
   const HTTP_REQUEST_TIMEOUT = 408; // The request was not acceptable
   const HTTP_TOO_MANY_REQUESTS = 429; // The request could not be completed due to a conflict with the current state of the resource
   const HTTP_INTERNAL_SERVER_ERROR = 500; // The server encountered an unexpected error
   const HTTP_BAD_GATEWAY = 502; // The server does not recognise the request method
   const HTTP_GATEWAY_TIMEOUT = 504;

   public function __construct()
   {
      parent::__construct();

      //Header
      $this->headers = $this->get_custom_header();
      //Custom Header

      //Router
      $this->ct_directory = $this->router->directory;
      $this->ct_ctrl = $this->router->directory.$this->router->fetch_class();
      $this->ct_action = $this->router->fetch_method();
      $this->ct_method = $this->input->server('REQUEST_METHOD');
   }

   /**
    * Not allow custom header
    */
   protected function get_custom_header()
   {
      $headers = $this->input->request_headers();

      return $headers;
   }

   /**
    * @return float
    */
   protected function get_execute_time()
   {
      $time = microtime();
      list($usec, $sec) = explode(' ', $time);
      $time = $usec + $sec;
      $total_time = number_format($time - LOAD_PAGE_START, 4);

      return $total_time;
   }

   /**
    *
    * Return Output Message
    *
    * @param null $data
    * @param string $message
    * @param int $status
    * @param bool $success
    */
   private function output(&$data = NULL, $message = '', $status = self::HTTP_OK, $success = FALSE)
   {
      $result = [
         'success' => $success,
         'data' => $data,
         'message' => $message,
         'time' => $this->get_execute_time()
      ];

      $this->output
         ->set_status_header($status)
         ->set_header('Cache-Control: private, max-age=0, no-cache, no-cache=Set-Cookie, proxy-revalidate')
         ->set_header("Pragma: no-cache")
         ->set_content_type('application/json')
         ->set_output(json_encode($result))
         ->_display();

      exit();
   }

   /**
    *
    * Return Success Message & Data
    *
    * @param $data
    * @param string $message
    */
   protected function success(&$data, $message = '', $status = self::HTTP_OK)
   {
      $this->output($data, $message, $status, TRUE);
   }

   /**
    *
    * Return Success Message
    *
    * @param string $message
    */
   protected function successN($message = '', $status = self::HTTP_OK)
   {
      $data = NULL;
      $this->output($data, $message, $status, TRUE);
   }

   /**
    *
    * Return Failed Message & Data
    *
    * @param $data
    * @param string $message
    */
   protected function failed(&$data, $message = 'Some thing failed.', $status = self::HTTP_OK)
   {
      $this->output($data, $message, $status, FALSE);
   }

   /**
    *
    * Return Failed Message
    *
    * @param string $message
    */
   protected function failedN($message = 'Some thing failed.', $status = self::HTTP_OK)
   {
      $data = NULL;
      $this->output($data, $message, $status, FALSE);
   }

   /**
    *
    * Return Failed Message & Data
    *
    * @param $data
    * @param string $message
    */
   protected function error(&$data, $message = 'Some thing error.', $status = self::HTTP_INTERNAL_SERVER_ERROR)
   {
      $this->output($data, $message, $status, FALSE);
   }

   /**
    *
    * Return Failed Message
    *
    * @param string $message
    */
   protected function errorN($message = 'Some thing error.', $status = self::HTTP_INTERNAL_SERVER_ERROR)
   {
      $data = NULL;
      $this->output($data, $message, $status, FALSE);
   }

}

// -------------- Auth Controller -------------- //
class Auth_Controller extends Base_Controller
{
   protected $login_user_id = FALSE;
   protected $login_user_role_id = FALSE;

   private $authorize = FALSE;
   protected $token = FALSE;
   protected $jwt_user = FALSE;

   public function __construct()
   {
      parent::__construct();
      $this->load->helpers('jwt');

      //Check Auth
      $this->authorization();
      $this->authentication();
   }

   /**
    * Get Header Authorization
    */
   protected function authorization()
   {
      if(isset($this->headers[$this->l_authorization]) && !empty($this->headers[$this->l_authorization])){
         $this->authorize = $this->headers[$this->l_authorization];
      }
      if(isset($this->headers[$this->u_authorization]) && !empty($this->headers[$this->u_authorization])){
         $this->authorize = $this->headers[$this->u_authorization];
      }
   }

   /**
    * Check User Authentication
    */
   protected function authentication()
   {
      //Check authorize and setup token
      if(!empty($this->authorize))
      {
         $tks = explode(' ', $this->authorize);

         if(!empty($tks[1])) {

            $this->token = trim($tks[1], '"');

            //Decode by JWT
            $salt = $this->config->item('jwt_encode_key');
            try
            {
               $this->jwt_user = JWT::decode($this->token, $salt);
            }
            catch(Exception $err){
               return $this->failedN($err->getMessage(), self::HTTP_UNAUTHORIZED);
            }

            $session_login = $this->session->userdata(LOGIN_SESSION_ADMIN);
            if(!empty($session_login))
            {
               //Check role
               if(!empty($session_login->role_id))
               {
                  list($restdir, $bos) = explode('/', $this->router->directory);
                  if($session_login->role_id && in_array($bos, ['bos', 'shared']))
                  {
                     //Admin access
                  }
                  else {
                     return $this->unauthorized_msg('1');
                  }
               }
               else {
                  return $this->unauthorized_msg('2');
               }

               $this->login_user_id = $session_login->user_id;
               $this->login_user_role_id = $session_login->role_id;

               return TRUE;
            }
            else {
               // check user from browser
               return $this->unauthorized_msg('3');
            }

         }

      }

      return $this->unauthorized_msg('4');
   }

   protected function unauthorized_msg($id)
   {
      return $this->failedN('Un authorized request. ('.$id.')', self::HTTP_UNAUTHORIZED);
   }

}

// -------------- Rest Controller -------------- //
class Restful_Controller extends Auth_Controller
{
   public $base_model = 'shared/base_model';
   protected $menu_ctl = FALSE;
   protected $allow_html_fields = [];
   protected $filters_listing = [];
   protected $filters_read = [];
   protected $filters_update = [];
   protected $filters_delete = [];
   protected $insert_data = [];
   protected $update_data = [];
   protected $primary_key = NULL;

   public function __construct()
   {
      parent::__construct();

      //Base model
      $this->load->model($this->base_model, 'model');

      $edit_actions = ['create', 'create_record', 'update', 'update_record', 'delete', 'delete_parent', 'reorder', 'status', 'highlight', 'reorder_item', 'reorder_rows'];
      $action = $this->ct_action = $this->router->fetch_method();
      if(IS_READONLY && in_array($action, $edit_actions) && $this->login_user_id != 1) {
         return $this->failedN('Read Only Mode');
      }
   }

   /**
    * Base : listing rows
    */
   public function listing()
   {
      if(!empty($this->input->get('filters')))
      {
         foreach($this->input->get('filters') as $key=>$val)
         {
            $this->filters_listing[$key] = input_to_sql($val);
         }
      }

      $listing = $this->model->listing($this->filters_listing);

      if(!empty($listing))
      {
         $primary_key = $this->primary_key;
         foreach($listing as $row)
         {
            $row->thumb_url = ASSETS_NOIMG_320;
            $row->cover_url = ASSETS_NOIMG_320;
            if(!empty($row->thumb_ext))
            {
               $row->thumb_url = base_url($this->thumb_config[0][0].$row->$primary_key.'.'.$row->thumb_ext.'?t='.time());
            }
            if(!empty($row->cover_ext))
            {
               $row->cover_url = base_url($this->cover_config[0][0].$row->$primary_key.'.'.$row->cover_ext.'?t='.time());
            }
            if(!empty($row->pdf_ext))
            {
               $row->pdf_url = base_url($this->pdf_config[0][0].$row->$primary_key.'.'.$row->pdf_ext.'?t='.time());
            }
         }
      }

      $res = new stdClass();
      $res->rows = $listing;
      $res->total_row = count($listing);
      $res->all_row = count($listing);

      return $this->success($res);
   }

   /**
    * Base : listing order by order_id
    */
   public function listing_order()
   {
      if(!empty($this->input->get('filters')))
      {
         foreach($this->input->get('filters') as $key=>$val)
         {
            $this->filters_listing[$key] = input_to_sql($val);
         }
      }

      $listing = $this->model->listing_order($this->filters_listing);

      if(!empty($listing))
      {
         $primary_key = $this->primary_key;
         foreach($listing as $row)
         {
            $row->thumb_url = ASSETS_NOIMG_320;
            $row->cover_url = ASSETS_NOIMG_320;
            if(!empty($row->thumb_ext))
            {
               $row->thumb_url = base_url($this->thumb_config[0][0].$row->$primary_key.'.'.$row->thumb_ext.'?t='.time());
            }
            if(!empty($row->cover_ext))
            {
               $row->cover_url = base_url($this->cover_config[0][0].$row->$primary_key.'.'.$row->cover_ext.'?t='.time());
            }
            if(!empty($row->pdf_ext))
            {
               $row->pdf_url = base_url($this->pdf_config[0][0].$row->$primary_key.'.'.$row->pdf_ext.'?t='.time());
            }
         }
      }

      $res = new stdClass();
      $res->rows = $listing;
      $res->total_row = count($listing);
      $res->all_row = count($listing);

      return $this->success($res);
   }

   /**
    * Base : read row
    */
   public function read($is_return = false)
   {
      $id = input_to_sql($this->input->post('id'));
      if(!empty($id))
      {
         $row = $this->model->read($id, $this->filters_read);

         if(!empty($row))
         {
            $row->thumb_url = ASSETS_NOIMG_320;
            $row->cover_url = ASSETS_NOIMG_320;
            $primary_key = $this->primary_key;
            if(!empty($row->thumb_ext))
            {
               $row->thumb_url = base_url($this->thumb_config[0][0].$row->$primary_key.'.'.$row->thumb_ext.'?t='.time());
            }
            if(!empty($row->cover_ext))
            {
               $row->cover_url = base_url($this->cover_config[0][0].$row->$primary_key.'.'.$row->cover_ext.'?t='.time());
            }
            if(!empty($row->special_ext))
            {
               $row->special_url = base_url($this->special_config[0][0].$row->$primary_key.'.'.$row->special_ext.'?t='.time());
            }
            if(!empty($row->pdf_ext))
            {
               $row->pdf_url = base_url($this->pdf_config[0][0].$row->$primary_key.'.'.$row->pdf_ext.'?t='.time());
            }

            if(!empty($row->detail_th)) {
               $row->detail_th = output_html($row->detail_th);
            }
            if(!empty($row->detail_en)) {
               $row->detail_en = output_html($row->detail_en);
            }
         }

         if($is_return) {
            return $row;
         }
         else {
            return $this->success($row);
         }
      }
      else{
         return $this->failedN('Not found id.');
      }
   }

   /**
    * Base : read row
    */
   public function read_lastest()
   {
      $row = $this->model->read_lastest($this->filters_read);

      if(!empty($row))
      {
         $row->thumb_url = ASSETS_NOIMG_320;
         $row->cover_url = ASSETS_NOIMG_320;
         $primary_key = $this->primary_key;
         if(!empty($row->thumb_ext))
         {
            $row->thumb_url = base_url($this->thumb_config[0][0].$row->$primary_key.'.'.$row->thumb_ext.'?t='.time());
         }
         if(!empty($row->cover_ext))
         {
            $row->cover_url = base_url($this->cover_config[0][0].$row->$primary_key.'.'.$row->cover_ext.'?t='.time());
         }
         if(!empty($row->pdf_ext))
         {
            $row->pdf_url = base_url($this->pdf_config[0][0].$row->$primary_key.'.'.$row->pdf_ext.'?t='.time());
         }

         if(!empty($row->detail_th)) {
            $row->detail_th = output_html($row->detail_th);
         }
         if(!empty($row->detail_en)) {
            $row->detail_en = output_html($row->detail_en);
         }

      }

      return $this->success($row);
   }

   /**
    * Base : delete rows
    */
   public function delete()
   {
      $ids = $this->input->post('ids');
      if(!empty($ids))
      {
         $this->model->delete($ids, $this->filters_delete);
         $this->_delete_cover();
         $this->_delete_pdf();
         $this->_delete_gallery();
         clear_main_cache();
         foreach($ids as $id) {
            $this->_delete_filters($id);
         }
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : delete by parent
    */
   public function delete_parent($parent_field, $parent_id)
   {
      $this->model->delete_parent($parent_field, $parent_id);
   }

   /**
    * Base : enable/disable rows
    */
   public function status()
   {
      $ids = $this->input->post('ids');
      $status = $this->input->post('status');
      if(!empty($ids))
      {
         $data = new stdClass();
         $data->status = $status;
         $data->created_by = $this->login_user_id;
         $this->model->status($ids, $data);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : highlight rows
    */
   public function highlight()
   {
      $ids = $this->input->post('ids');
      $highlight = $this->input->post('highlight');
      if(!empty($ids))
      {
         $data = new stdClass();
         $data->created_by = $this->login_user_id;
         $this->model->highlight($ids, $highlight, $data);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : re-order rows
    */
   public function reorder()
   {
      $ids = $this->input->post('ids');
      if(!empty($ids))
      {
         $data = new stdClass();
         $data->created_by = $this->login_user_id;
         $this->model->reorder($ids, $data);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : re-order / delete item
    */
   public function reorder_item()
   {
      $ids = $this->input->post('ids');
      $del = $this->input->post('del');
      if(!empty($ids))
      {
         $data = new stdClass();
         $data->created_by = $this->login_user_id;
         $this->model->reorder_item($ids, $data, $del);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : re-order rows
    */
   public function reorder_rows()
   {
      $orderids = $this->input->post('orderids');
      if(!empty($orderids))
      {
         $data = new stdClass();
         $data->created_by = $this->login_user_id;
         $this->model->reorder_rows($orderids, $data);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : re-order rows
    */
   public function reorder_change()
   {
      $orderids = $this->input->post('orderids');
      $current_id = $this->input->post('current_id');
      if(!empty($orderids))
      {
         $data = new stdClass();
         $data->created_by = $this->login_user_id;
         $this->model->reorder_change($orderids, $data, $current_id);
         clear_main_cache();
      }
      else{
         return $this->failedN('Not found ids.');
      }
   }

   /**
    * Base : create row
    */
   public function create()
   {
      $result = $this->create_record();
      if($result)
      {
         return $this->success($result);
      }
      else{
         return $this->failedN('Cannot create row.');
      }
   }

   public function create_record()
   {
      $data = new stdClass();
      if(isset($_POST) && !empty($_POST))
      {
         foreach($_POST as $key => $val)
         {
            if(!empty($val) && !is_array($val))
            {
               $data->$key = input_to_sql($val, in_array($key, $this->allow_html_fields));
            }
            else if(!is_array($val)){
               $data->$key = null;
            }
         }
      }

      if(!empty($this->insert_data))
      {
         foreach($this->insert_data as $key=>$val)
         {
            if(!empty($val)){
               $data->$key = input_to_sql($val);
            }
         }
      }

      if(!empty($_FILES['cover']['name']))
      {
         $data->cover_ext = file_ext($_FILES['cover']['name']);
      }

      if(!empty($_FILES['thumb']['name']))
      {
         $data->thumb_ext = file_ext($_FILES['thumb']['name']);
      }

      if(!empty($_FILES['special']['name']))
      {
         $data->special_ext = file_ext($_FILES['special']['name']);
      }

      if(!empty($_FILES['pdf']['name']))
      {
         $data->pdf_ext = file_ext($_FILES['pdf']['name']);
      }

      if(!empty($data))
      {
         $data->created_by = $this->login_user_id;
         $id = $this->model->create($data);
         if($id)
         {
            $this->_upload_cover($id);
            $this->_upload_pdf($id);
            $this->_upload_gallery($id, $data);
            $this->_create_filters($id);
            return $id;
         }
         else {
            return FALSE;
         }
      }
      else{
         return $this->failedN('Parameter not found.');
      }
   }

   /**
    * Base : update row
    */
   public function update()
   {
      $result = $this->update_record();
      if($result)
      {
         return $this->success($result);
      }
      else{
         return $this->failedN('Cannot update row.');
      }
   }

   public function update_record()
   {
      $data = new stdClass();
      if(isset($_POST) && !empty($_POST))
      {
         foreach($_POST as $key => $val)
         {
            if(!empty($val) && !is_array($val))
            {
               $data->$key = input_to_sql($val, in_array($key, $this->allow_html_fields));
            }
            else if(!is_array($val)){
               if($key == 'pageview')
               {
                  $data->$key = 0;
               }
               else {
                  $data->$key = null;
               }
            }
         }
      }

      if(!empty($this->update_data))
      {
         foreach($this->update_data as $key=>$val)
         {
            if(!empty($val)){
               $data->$key = input_to_sql($val);
            }
         }
      }

      if(!empty($_FILES['cover']['name']))
      {
         $data->cover_ext = file_ext($_FILES['cover']['name']);
      }
      $delete_cover = NULL;
      if(!empty($data->delete_cover)){
         $delete_cover = $data->delete_cover;
         $data->cover_ext = NULL;
         unset($data->delete_cover);
      }

      if(!empty($_FILES['thumb']['name']))
      {
         $data->thumb_ext = file_ext($_FILES['thumb']['name']);
      }
      $delete_thumb = NULL;
      if(!empty($data->delete_thumb)){
         $delete_thumb = $data->delete_thumb;
         $data->thumb_ext = NULL;
         unset($data->delete_thumb);
      }

      if(!empty($_FILES['special']['name']))
      {
         $data->special_ext = file_ext($_FILES['special']['name']);
      }
      $delete_special = NULL;
      if(!empty($data->delete_special)){
         $delete_special = $data->delete_special;
         $data->special_ext = NULL;
         unset($data->delete_special);
      }

      if(!empty($_FILES['pdf']['name']))
      {
         $data->pdf_ext = file_ext($_FILES['pdf']['name']);
      }
      $delete_pdf = NULL;
      if(!empty($data->delete_pdf)){
         $delete_pdf = $data->delete_pdf;
         $data->pdf_ext = NULL;
         unset($data->delete_pdf);
      }

      $delete_gallery = NULL;
      if(!empty($this->input->post('delete_gallery'))){
         $delete_gallery = $this->input->post('delete_gallery');
      }

      if(!empty($data))
      {
         $data->created_by = $this->login_user_id;
         $id = $this->model->update($data);
         if($id)
         {
            $this->_upload_cover($id); // $result = $id
            $this->_delete_cover_byid($id, $delete_cover);
            $this->_delete_thumb_byid($id, $delete_thumb);
            $this->_delete_special_byid($id, $delete_special);
            $this->_upload_pdf($id);
            $this->_delete_pdf_byid($id, $delete_pdf);
            $this->_upload_gallery($id, $data);
            $this->_update_gallery();
            $this->_delete_gallery_byid($id, $delete_gallery);
            $this->_create_filters($id);
            return $id;
         }
         else{
            return FALSE;
         }
      }
      else{
         return $this->failedN('Parameter not found.');
      }
   }

   /**
    * Check existing row or field
    */
   public function existing()
   {
      $field = $this->input->post('field');
      $val = $this->input->post('val');
      $exclude = $this->input->post('exclude');

      if(!empty($field) && !empty($val))
      {
         if($this->model->existing($field, $val, $exclude))
         {
            return $this->failedN('This '.$field.': '.$val.' already exists');
         }
      }
      else{
         return $this->failedN('Parameter not found.');
      }
   }

   /**
    * Cover Upload Management
    */
   protected $cover_config = [];
   protected $thumb_config = [];
   protected $special_config = [];

   protected function _upload_cover($id)
   {
      $this->load->model('shared/upload_model');

      if(!empty($_FILES['cover']['name']) || !empty($_FILES['thumb']['name']) || !empty($_FILES['special']['name']))
      {
         if(!empty($_FILES['cover']['name']))
         {
            $this->_remove_cover_file($id);
            $this->upload_model->upload_cover($id, $this->cover_config);
         }

         if(!empty($_FILES['thumb']['name']))
         {
            $this->_remove_thumb_file($id);
            $this->upload_model->upload_thumb($id, $this->thumb_config);
         }

         if(!empty($_FILES['special']['name']))
         {
            $this->_remove_special_file($id);
            $this->upload_model->upload_special($id, $this->special_config);
         }
      }
   }

   protected function _delete_cover()
   {
      $ids = $this->input->post('ids');
      if(!empty($ids) && !empty($this->cover_config))
      {
         foreach($ids as $id)
         {
            $this->_remove_cover_file($id);
         }
      }

      if(!empty($ids) && !empty($this->thumb_config))
      {
         foreach($ids as $id)
         {
            $this->_remove_thumb_file($id);
         }
      }

      if(!empty($ids) && !empty($this->special_config))
      {
         foreach($ids as $id)
         {
            $this->_remove_special_file($id);
         }
      }
   }

   protected function _delete_cover_byid($id, $delete_cover = '')
   {
      if(!empty($delete_cover))
      {
         $this->_remove_cover_file($id);
      }
   }

   protected function _remove_cover_file($id)
   {
      if(!empty($this->cover_config))
      {
         foreach($this->cover_config as $item)
         {
            if(!empty($item[0]))
            {
               $path = $item[0];
               foreach(UPLOAD_EXTENSION as $ext)
               {
                  @unlink($path.$id.'.'.$ext);
               }
            }
         }
      }
   }

   protected function _delete_thumb_byid($id, $delete_thumb = '')
   {
      if(!empty($delete_thumb))
      {
         $this->_remove_thumb_file($id);
      }
   }

   protected function _remove_thumb_file($id)
   {
      if(!empty($this->thumb_config))
      {
         foreach($this->thumb_config as $item)
         {
            if(!empty($item[0]))
            {
               $path = $item[0];
               foreach(UPLOAD_EXTENSION as $ext)
               {
                  @unlink($path.$id.'.'.$ext);
               }
            }
         }
      }
   }

   protected function _delete_special_byid($id, $delete_special = '')
   {
      if(!empty($delete_special))
      {
         $this->_remove_special_file($id);
      }
   }

   protected function _remove_special_file($id)
   {
      if(!empty($this->special_config))
      {
         foreach($this->special_config as $item)
         {
            if(!empty($item[0]))
            {
               $path = $item[0];
               foreach(UPLOAD_EXTENSION as $ext)
               {
                  @unlink($path.$id.'.'.$ext);
               }
            }
         }
      }
   }

   /**
    * PDF Upload Management
    */
   protected $pdf_config = [];

   protected function _upload_pdf($id)
   {
      $this->load->model('shared/upload_model');

      if(!empty($_FILES['pdf']['name']))
      {
         $this->_remove_pdf_file($id);
         $this->upload_model->upload_pdf($id, $this->pdf_config);
      }
   }

   protected function _delete_pdf()
   {
      $ids = $this->input->post('ids');
      if(!empty($ids) && !empty($this->pdf_config))
      {
         foreach($ids as $id)
         {
            $this->_remove_pdf_file($id);
         }
      }
   }

   protected function _delete_pdf_byid($id, $delete_pdf = '')
   {
      if(!empty($delete_pdf))
      {
         $this->_remove_pdf_file($id);
      }
   }

   protected function _remove_pdf_file($id)
   {
      if(!empty($this->pdf_config))
      {
         foreach($this->pdf_config as $item)
         {
            if(!empty($item[0]))
            {
               $path = $item[0];
               @unlink($path.$id.'.pdf');
            }
         }
      }
   }

   /**
    * Gallery Upload Management
    */
   protected $gallery_config = [];
   protected $gallery_model = NULL;
   protected $gallery_parent_key = NULL;
   protected function _upload_gallery($id, $data)
   {
      if(!empty($_FILES['gallery']['name']))
      {
         $this->load->model('shared/upload_model');
         $this->gallery_config[0][0] = sprintf($this->gallery_config[0][0], $id);
         $this->gallery_config[1][0] = sprintf($this->gallery_config[1][0], $id);
         $data->created_by = $this->login_user_id;
         $this->upload_model->upload_gallery($id, $this->gallery_config, $this->gallery_model, $this->gallery_parent_key, $data);
      }
   }

   protected function _delete_gallery()
   {
      $parent_ids = $this->input->post('ids');
      if(!empty($parent_ids) && !empty($this->gallery_model))
      {
         $this->load->model($this->gallery_model, 'gmodel');
         $delete_gallery_ids = [];
         foreach($parent_ids as $parent_id)
         {
            $filters[$this->gallery_parent_key] = $parent_id;
            $listing = $this->gmodel->listing($filters);
            if(!empty($listing))
            {
               foreach($listing as $row)
               {
                  $delete_gallery_ids[] = $row->gallery_id;
               }
            }
            $this->_delete_gallery_byid($parent_id, $delete_gallery_ids);
         }
      }
   }

   protected function _delete_gallery_byid($parent_id, $delete_gallery_ids = [])
   {
      if(!empty($parent_id) && !empty($delete_gallery_ids))
      {
         foreach($delete_gallery_ids as $gallery_id)
         {
            $this->_remove_gallery_file($parent_id, $gallery_id);
         }
         $this->gmodel->delete($delete_gallery_ids);
      }
   }

   protected function _remove_gallery_file($parent_id, $gallery_id)
   {
      if(!empty($this->gallery_config))
      {
         foreach($this->gallery_config as $items)
         {
            if(!empty($items))
            {
               $folder = $items[0];
               $path = sprintf($folder, $parent_id);
               foreach(UPLOAD_EXTENSION as $ext)
               {
                  @unlink($path.$gallery_id.'.'.$ext);
               }
            }
         }
      }
   }

   protected function _update_gallery()
   {
      $current_gallery_id = $this->input->post('current_gallery_id');
      if(!empty($current_gallery_id))
      {
         $current_gallery_orderid = $this->input->post('current_gallery_orderid');
         $current_gallery_title = $this->input->post('current_gallery_title');
         $current_gallery_alt = $this->input->post('current_gallery_alt');

         $this->load->model($this->gallery_model, 'gmodel');
         foreach ($current_gallery_id as $gallery_id)
         {
            $row = new stdClass();
            $row->gallery_id = $gallery_id;
            $row->orderid = $current_gallery_orderid[$gallery_id];
            $row->cover_title = $current_gallery_title[$gallery_id];
            $row->cover_alt = $current_gallery_alt[$gallery_id];
            $row->created_by = $this->login_user_id;
            $this->gmodel->update($row);
         }
      }
   }


   /**
    * Filters Management
    */
   protected $filters_item_model = NULL;
   protected $filters_parent_key = NULL;
   protected $filters_map_key = NULL;
   protected function _create_filters($parent_id)
   {
      $current_filters = $this->input->post('filters');
      $filter_detail = $this->input->post('filter_detail');
      if(!empty($current_filters))
      {
         $parent_key = $this->filters_parent_key;
         $map_key = $this->filters_map_key;
         $this->load->model($this->filters_item_model, 'ftmodel');

         $this->_delete_filters($parent_id);

         foreach ($current_filters as $map_id)
         {
            $row = new stdClass();
            $row->$parent_key = $parent_id;
            $row->$map_key = $map_id;
            $row->detail = $filter_detail[$map_id];
            $row->status = GLOBAL_STATUS_ENABLED;
            $row->created_by = $this->login_user_id;
            $this->ftmodel->create($row);
         }
      }
   }

   protected function _delete_filters($parent_id)
   {
      $parent_key = $this->filters_parent_key;
      if(!empty($parent_key)) {
         $this->load->model($this->filters_item_model, 'ftmodel');
         $this->ftmodel->delete_parent($parent_key, $parent_id);
      }
   }

   /**
    * File Upload Management
    */
   protected $file_config = [];
   protected $file_model = NULL;
   protected $file_parent_key = NULL;
   protected function _upload_file($id, $data)
   {
      if(!empty($_FILES['file']['name']))
      {
         $this->load->model('shared/upload_model');
         $data->created_by = $this->login_user_id;
         $this->upload_model->upload_files($id, $this->file_config, $this->file_model, $this->file_parent_key, $data);
      }
   }

   protected function _delete_file()
   {
      $parent_ids = $this->input->post('ids');
      if(!empty($parent_ids) && !empty($this->file_model))
      {
         $this->load->model($this->file_model, 'fmodel');
         $delete_file_ids = [];
         foreach($parent_ids as $parent_id)
         {
            $filters[$this->file_parent_key] = $parent_id;
            if(!empty($listing))
            {
               foreach($listing as $row)
               {
                  $delete_file_ids[] = $row->file_id;
               }
            }
            $this->_delete_file_byid($parent_id, $delete_file_ids);
         }
      }
   }

   protected function _delete_file_byid($parent_id, $delete_file_ids = [])
   {
      if(!empty($parent_id) && !empty($delete_file_ids))
      {
         foreach($delete_file_ids as $file_id)
         {
            $this->_remove_file_file($parent_id, $file_id);
         }
         $this->fmodel->delete($delete_file_ids);
      }
   }

   protected function _remove_file_file($parent_id, $file_id)
   {
      if(!empty($this->file_config))
      {
         foreach($this->file_config as $items)
         {
            if(!empty($items))
            {
               $folder = $items[0];
               $path = sprintf($folder, $parent_id);
               foreach(UPLOAD_FILE_EXTENSION as $ext)
               {
                  @unlink($path.$file_id.'.'.$ext);
               }
            }
         }
      }
   }

   protected function _update_file()
   {
      $current_file_id = $this->input->post('current_file_id');
      if(!empty($current_file_id))
      {
         $current_file_orderid = $this->input->post('current_file_orderid');
         $current_file_title = $this->input->post('current_file_title');
         $current_file_alt = $this->input->post('current_file_alt');

         $this->load->model($this->file_model, 'fmodel');
         foreach ($current_file_id as $file_id)
         {
            $row = new stdClass();
            $row->file_id = $file_id;
            $row->orderid = $current_file_orderid[$file_id];
            $row->file_title = $current_file_title[$file_id];
            $row->file_alt = $current_file_alt[$file_id];
            $row->created_by = $this->login_user_id;
            $this->fmodel->update($row);
         }
      }
   }
}

// -------------- Admin Controller -------------- //
class Admin_Controller extends CI_Controller
{
   //Main
   public $comgroup;
   public $component;
   public $action;
   public $action_page = 'listing'; // listing, view, add, edit, items
   public $component_title = '';

   //Menu
   public $com_title = '';
   public $menu_group_id;
   public $menu_id;

   //Listing
   public $row_per_page = 5;

   public $session_login = FALSE;
   public $is_logged_in = FALSE;
   public $user_menu = FALSE;
   public $user_id = FALSE;
   public $user_name = FALSE;
   public $user_email = FALSE;
   public $user_avatar = FALSE;
   public $user_gender = FALSE;
   public $role_id = FALSE;
   public $role_name = FALSE;
   public $settings = FALSE;

   //Permission
   public $permissions = FALSE;
   public $component_icon = FALSE;
   public $permits = FALSE;
   public $is_listing = FALSE;
   public $is_view = FALSE;
   public $is_add = FALSE;
   public $is_edit = FALSE;
   public $is_publish = FALSE;
   public $is_delete = FALSE;
   public $is_highlight = FALSE;
   public $allow_permission = FALSE;
   public $hide_reorder = FALSE;

   //Menu
   public $menu_group_names = [];
   public $menu_names = [];
   public $menu_ids = [];

   public function __construct()
   {
      parent::__construct();

      if($this->session_login = $this->session->userdata(LOGIN_SESSION_ADMIN))
      {
         $this->is_logged_in = TRUE;
         $this->user_id = $this->session_login->user_id;
         $this->user_name = $this->session_login->firstname;
         $this->user_email = $this->session_login->email;
         $this->user_avatar = $this->session_login->cover_ext;
         $this->user_gender = $this->session_login->gender;
         $this->user_menu = $this->session_login->menu;
         $this->menu_group_names = $this->session_login->menu_group_names;
         $this->menu_names = $this->session_login->menu_names;
         $this->role_id = $this->session_login->role_id;
         $this->role_name = $this->session_login->role_name;
         $this->settings = $this->session_login->settings;
         $this->menu_ids = $this->session_login->menu_ids;

         //view
         if(!empty($this->uri->segment(4))){
            $this->comgroup = $this->uri->segment(2);
            $this->component = $this->uri->segment(3);
            $this->action_page = $this->uri->segment(4);
         }
         else if(!empty($this->uri->segment(3))){
            $this->comgroup = $this->uri->segment(2);
            $this->component = $this->uri->segment(3);
            if(in_array($this->uri->segment(3), ['add', 'edit', 'view']))
            {
               $this->component = $this->uri->segment(2);
               $this->action_page = $this->uri->segment(3);
            }
         }
         //rest
         else{
            $this->comgroup = $this->uri->segment(2);
            $this->component = $this->uri->segment(2);
         }

         $this->component_title = $this->component_name();
         $this->menu_id = $this->menu_ids[$this->component];

         $this->permission_assign();
         $this->allow_permission();

      }
      else{
         redirect(base_url('bos/login'));exit();
      }
   }

   private function component_name()
   {
      $title = '';
      $exp = explode('_', $this->component);
      foreach($exp as $e){
         $title .= ucfirst($e).' ';
      }

      return $title;
   }

   /**
    * Check Permission Menu
    */
   private function permissions()
   {
      $menu = [];
      $icon = [];
      $permits = [];
      $menus = $this->user_menu;
      if(!empty($menus))
      {
         foreach($menus as $menu_group)
         {
            $flag = $menu_group['group']['flag'];
            $items = $menu_group['group']['items'];
            $icon[$flag] = $menu_group['group']['icon'];

            if(!empty($items))
            {
               foreach($items as $item)
               {
                  $menu[] = $item['flag'];
                  $permits[$item['flag']] = $item['permits'];
               }
            }
         }
      }

      return [$menu, $icon, $permits];
   }

   private function permission_assign()
   {
      $permission = $this->permissions();
      $this->permissions = isset($permission[0]) ? $permission[0]:FALSE;
      $this->component_icon = isset($permission[1]) ? $permission[1]:FALSE;
      $this->permits = isset($permission[2]) ? $permission[2]:FALSE;

      if(!empty($this->permits[$this->component]))
      {
         $this->is_listing = $this->permits[$this->component]['listing'];
         $this->is_view = $this->permits[$this->component]['view'];
         $this->is_add = $this->permits[$this->component]['add'];
         $this->is_edit = $this->permits[$this->component]['edit'];
         $this->is_publish = $this->permits[$this->component]['publish'];
         $this->is_delete = $this->permits[$this->component]['delete'];
         $this->is_highlight = $this->permits[$this->component]['highlight'];
      }
   }

   private function allow_permission()
   {
      if($this->action_page == 'listing' && $this->is_listing){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'view' && $this->is_view){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'add' && $this->is_add){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'edit' && $this->is_edit){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'publish' && $this->is_publish){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'delete' && $this->is_delete){
         $this->allow_permission = TRUE;
      }
      else if($this->action_page == 'highlight' && $this->is_highlight){
         $this->allow_permission = TRUE;
      }
      else{
         $this->allow_permission = FALSE;
      }
   }

}

// -------------- Admin View Controller -------------- //
class AdminView_Controller extends Admin_Controller
{
   protected $baseData = [];
   private $ct_directory;
   private $ct_ctrl;
   private $ct_action;

   //Form
   protected $view_path = '';
   protected $view_data = [];
   protected $add_path = '';
   protected $edit_path = '';

   protected $is_search_btn = TRUE;
   protected $custom_data = [];
   protected $folder = 'bos';
   protected $is_ckeditor = FALSE;

   protected $form_inline = FALSE;

   protected $is_multiple_checkbox = FALSE;

   public function __construct()
   {
      parent::__construct();

      $this->ct_directory = $this->router->directory;
      $this->ct_ctrl = $this->router->directory.$this->router->fetch_class();
      $this->ct_action = $this->router->fetch_method();

      $this->view_path = $this->folder.'/'.$this->component.'/view';
      $this->add_path = $this->folder.'/'.$this->component.'/add';
      $this->edit_path = $this->folder.'/'.$this->component.'/edit';

      $base_data = [
         'directory' => $this->ct_directory,
         'controller' => $this->ct_ctrl,
         'ct_action' => $this->ct_action,
         'com_file' => $this->component.'/'.$this->action_page,
         'comgroup' => $this->comgroup,
         'component' => $this->component,
         'component_title' => $this->component_title,
         'component_icon' => isset($this->component_icon[$this->comgroup]) ? $this->component_icon[$this->comgroup]:'',
         'action_page' => $this->action_page,
         'permissions' => $this->permissions,
         'permits' => $this->permits,
         'allow_permission' => $this->allow_permission,
         'user_menu' => $this->user_menu,
         'menu_group_name' => $this->menu_group_names[$this->comgroup],
         'menu_name' => $this->menu_names[$this->component],
         'user_id' => $this->user_id,
         'user_name' => $this->user_name,
         'user_email' => $this->user_email,
         'user_avatar' => $this->user_avatar,
         'user_gender' => $this->user_gender,
         'role_id' => $this->role_id,
         'role_name' => $this->role_name,
         'settings' => $this->settings,
         'is_listing' => $this->is_listing,
         'is_view' => $this->is_view,
         'is_add' => $this->is_add,
         'is_edit' => $this->is_edit,
         'is_publish' => $this->is_publish,
         'is_highlight' => $this->is_highlight,
         'is_ckeditor' => $this->is_ckeditor,
         'button_main' => $this->button_main(),
         'button_row' => $this->button_row(),
         'button_inline' => $this->button_row(),
         'menu_id' => $this->menu_id
      ];

      $this->baseData = $base_data;

      if(strpos($this->ct_directory, $this->folder) !== FALSE && $this->role_id)
      {
         $this->folder = 'bos';
      }
      else {
         exit('Permission Denied !');
      }
   }

   /**
    * Main
    */
   protected function main()
   {
      if(!empty($this->custom_data)){
         $this->baseData = array_merge($this->baseData, $this->custom_data);
      }

      $this->load->view($this->folder.'/main', $this->baseData);
   }

   /**
    * Display
    */
   public function index()
   {
      $this->main();
   }

   /**
    * View
    */
   public function view()
   {
      if($this->is_view == 1)
      {
         if($this->form_inline)
         {
            $this->load->view($this->folder.'/main', $this->baseData);
         }
         else {
            $this->load->view($this->view_path);
         }
      }
   }

   /**
    * Form Add
    */
   public function add()
   {
      if($this->is_add == 1)
      {
         if($form = $this->input->get('form'))
         {
            $this->add_path = $this->add_path.'-'.$form.'.php';
         }

         if($this->form_inline)
         {
            $this->load->view($this->folder.'/main', $this->baseData);
         }
         else {
            $this->load->view($this->add_path, $this->baseData);
         }
      }
   }

   /**
    * Form Edit
    */
   public function edit()
   {
      if($this->is_edit == 1)
      {
         if($this->form_inline)
         {
            $this->load->view($this->folder.'/main', $this->baseData);
         }
         else {
            $this->load->view($this->edit_path, $this->baseData);
         }
      }
   }

   /**
    * Button
    */
   private function button_main()
   {
      $default_text = [
         'add' => 'เพิ่มรายการใหม่',
         'edit' => 'แก้ไขรายการที่เลือก',
         'delete' => 'ลบรายการที่เลือกไว้',
         'action' => 'ดำเนินการ',
         'highlight' => 'ทำไฮไลท์',
         'unhighlight' => 'ยกเลิกไฮไลท์',
         'reorder' => 'จัดลำดับไฮไลท์',
         'enable' => 'ใช้งาน',
         'disable' => 'ระงับ'
      ];

      //contact
      if($this->ct_ctrl == 'bos/contact')
      {
         $default_text['enable'] = 'อ่านแล้ว';
         $default_text['disable'] = 'ยังไม่อ่าน';
      }
      if($this->ct_ctrl == 'bos/user_logs')
      {
         $default_text['enable'] = 'สำเร็จ';
         $default_text['disable'] = 'ไม่สำเร็จ';
      }

      if(!empty($custom_text))
      {
         foreach($custom_text as $key=>$val)
         {
            $default_text[$key] = $val;
         }
      }

      $btn_merge = [];

      //inline
      if($this->is_add)
      {
         $add_new_class = 'btn-addnew';
         if($this->form_inline)
         {
            $add_new_class = 'btn-addnew-inline';
         }
         $btn_merge[] = '<button type="button" class="btn btn-primary btn-sm shadow '.$add_new_class.'" title="'.$default_text['add'].'"><i class="la la-plus"></i> '.$default_text['add'].'</button>';
      }
      //if(!$this->is_multiple_checkbox && $this->is_highlight && !$this->hide_reorder)
      //{
      //   $btn_merge[] = '<button type="button" class="btn btn-light btn-sm shadow toolbar-main-reorder" title="'//.$default_text['reorder'].'"><i class="la la-sort"></i> '.$default_text['reorder'].'</button>';
      //}

      //dropdown
      $btn_group = [];
      if($this->is_publish)
      {
         $btn_group[] = '<button class="dropdown-item toolbar-main-enable" type="button" title="'.$default_text['enable'].'"><i class="la la-check-circle text-success"></i> '.$default_text['enable'].'</button>
            <button class="dropdown-item toolbar-main-disable" type="button" title="'.$default_text['disable'].'"><i class="la la-minus-circle text-danger"></i> '.$default_text['disable'].'</button>';
      }

      if($this->is_publish && $this->is_delete)
      {
         $btn_group[] = '<div class="dropdown-divider"></div>';
      }

      if($this->is_delete)
      {
         $btn_group[] = '<button class="dropdown-item toolbar-main-delete" type="button" title="'.$default_text['delete'].'"><i class="la la-trash"></i> '.$default_text['delete'].'</button>';
      }

      if($this->is_highlight)
      {
         $btn_group[] = '<div class="dropdown-divider"></div>
            <button class="dropdown-item toolbar-main-highlight" type="button" title="'.$default_text['highlight'].'"><i class="las la-star text-warning"></i> '.$default_text['highlight'].'</button>
            <button class="dropdown-item toolbar-main-unhighlight" type="button" title="'.$default_text['unhighlight'].'"><i class="lar la-star"></i> '.$default_text['unhighlight'].'</button>
            <button class="dropdown-item toolbar-main-reorder" type="button" title="'.$default_text['reorder'].'"><i class="las la-sort"></i> '.$default_text['reorder'].'</button>';
      }

      if(!empty($btn_group) && $this->is_multiple_checkbox)
      {
         $btn_merge[] = '<div class="btn-group">
         <button type="button" class="btn btn-light btn-sm dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false" title="'.$default_text['action'].'">
            '.$default_text['action'].' <i class="la la-caret-down invisible"></i>
         </button>
         <div class="dropdown-menu dropdown-menu-right dropdown-menu-lg-left">
            '.implode(' ', $btn_group).'
         </div>
      </div>';
      }

      return implode(' ', $btn_merge);
   }

   private function button_row()
   {
      $btn = '';

      if($this->is_view)
      {
         $view_new_class = 'toolbar-row-view';
         if($this->form_inline)
         {
            $view_new_class = 'toolbar-row-view-inline';
         }
         $btn .= '<button class="btn btn-sm btn-light shadow '.$view_new_class.'" type="button" title="ดูรายละเอียดรายการนี้"><i class="las la-eye align-middle"></i></button>';
      }

      if($this->is_edit)
      {
         $edit_new_class = 'toolbar-row-edit';
         if($this->form_inline)
         {
            $edit_new_class = 'toolbar-row-edit-inline';
         }
         $btn .= '<button class="btn btn-sm btn-light shadow '.$edit_new_class.'" type="button" title="แก้ไขรายการนี้"><i class="las la-pen align-middle"></i></button>';
      }

      if($this->is_delete)
      {
         $btn .= '<button class="btn btn-light btn-sm shadow toolbar-row-delete" type="button" title="ลบรายการนี้"><i class="la la-trash align-middle"></i></button>';
      }

      return $btn;

   }

}

// -------------- Frontend Rest Controller -------------- //
class Frontend_Rest extends Auth_Controller
{
   public $current_lang = DEFAULT_LANG;
   protected $main_url = NULL;
   protected $base_url = NULL;
   protected $current_url = NULL;
   protected $current_th_url = NULL;
   protected $current_en_url = NULL;

   public function __construct()
   {
      parent::__construct();

      $this->current_lang();
      $this->current_lang_url($this->current_lang);

      $this->main_url = base_url();
      $this->base_url = base_url($this->current_lang != DEFAULT_LANG ? $this->current_lang.'/':'');
      $this->current_url = current_url();
      $this->current_th_url = $this->current_lang_url('th');
      $this->current_en_url = $this->current_lang_url('en');
   }

   protected function authentication()
   {
      return TRUE;
   }

   private function current_lang()
   {
      $lang = $this->uri->segment(1);
      if(!empty($lang) && in_array($lang, AVAILABLE_LANG))
      {
         $this->current_lang = $lang;
      }
      else {
         $this->current_lang = DEFAULT_LANG;
      }

      $lang = $this->input->post('lang');
      if(!empty($lang))
      {
         $this->current_lang = $lang;
      }

      $this->load->helper('lang/'.$this->current_lang);
   }

   private function current_lang_url($lang)
   {
      $uri_string = $this->uri->uri_string();

      if(in_array($this->uri->segment(1), AVAILABLE_LANG))
      {
         if($lang != DEFAULT_LANG)
         {
            $uri_string = str_replace($lang.'/', '/', $uri_string);
            $url = $lang.$uri_string;
         }
         else
         {
            $uri_string = str_replace($this->uri->segment(1).'/', '/', $uri_string);
            $url = $uri_string;
         }
      }
      else
      {
         if($lang != DEFAULT_LANG)
         {
            $url = $lang.'/'.$uri_string;
         }
         else
         {
            $url = $uri_string;
         }
      }

      return base_url($url);
   }
}

// -------------- Frontend View Controller -------------- //
class Frontend_View extends CI_Controller
{
   public $current_lang = DEFAULT_LANG;
   protected $row = [];
   protected $thumb_url = NULL;
   protected $base_url = NULL;
   protected $meta_title = SETTING_SEO_META_TITLE;
   protected $meta_description = SETTING_SEO_META_DESCRIPTION;
   protected $meta_keywords = SETTING_SEO_META_KEYWORDS;
   protected $meta_published_time = NULL;

   public function __construct()
   {
      parent::__construct();

      $this->current_lang();

      $this->base_url = base_url($this->current_lang != DEFAULT_LANG ? $this->current_lang.'/':'');
   }

   private function current_lang()
   {
      $lang = $this->uri->segment(1);
      if(!empty($lang) && in_array($lang, AVAILABLE_LANG))
      {
         $this->current_lang = $lang;
      }
      else {
         $this->current_lang = DEFAULT_LANG;
      }

      $lang = $this->input->post('lang');
      if(!empty($lang))
      {
         $this->current_lang = $lang;
      }

      $this->load->helper('lang/'.$this->current_lang);
   }

   private function current_lang_url($lang)
   {
      $uri_string = $this->uri->uri_string();

      if(in_array($this->uri->segment(1), AVAILABLE_LANG))
      {
         if($lang != DEFAULT_LANG)
         {
            $uri_string = str_replace($lang.'/', '/', $uri_string);
            $url = $lang.$uri_string;
         }
         else
         {
            $uri_string = str_replace($this->uri->segment(1).'/', '/', $uri_string);
            $url = $uri_string;
         }
      }
      else
      {
         if($lang != DEFAULT_LANG)
         {
            $url = $lang.'/'.$uri_string;
         }
         else
         {
            $url = $uri_string;
         }
      }

      return base_url($url);
   }

   public function view($path, $data = [], $html = FALSE)
   {
      $this->output->cache(SETTING_ADMIN_CACHE);

      //global data
      $data['current_lang'] = $this->current_lang;
      $data['current_view_folder'] = 'application/views/front/'.$this->current_lang.'/';
      $data['main_url'] = base_url();
      $data['base_url'] = base_url($this->current_lang != DEFAULT_LANG ? $this->current_lang.'/':'');
      $data['current_url'] = current_url();
      $data['current_th_url'] = $this->current_lang_url('th');
      $data['current_en_url'] = $this->current_lang_url('en');
      $data['theme_url'] = base_url('theme/');
      $data['captcha_url'] = base_url('captcha/display');

      //default
      $this->load->model('front/product_category_model');
      $data['product_category_listing'] = $this->product_category_model->listing_order();

      //cart
      $qty = 0;
      $current_cart = get_cookie(CART_COOKIE);
      if(!empty($current_cart)) {
         $current_item = json_decode($current_cart);
         $qty = count($current_item);
      }
      $data['qty'] = $qty;

      //meta tag
      $data = array_merge($data, $this->meta_tags());

      $get_html = $this->load->view('front/'.$this->current_lang.'/'.$path, $data, $html);
      if($html){
         return $get_html;
      }
   }

   private function meta_tags()
   {
      $meta_data = [];

      $meta_data['meta_title'] = $this->meta_title;
      $meta_data['meta_description'] = $this->meta_description;
      $meta_data['meta_keywords'] = $this->meta_keywords;
      $meta_data['meta_published_time'] = $this->meta_published_time;

      $meta_image_url = NULL;
      if(!empty($this->thumb_url))
      {
         $meta_image_url = $this->thumb_url;
         list($meta_image_width, $meta_image_height) = getimagesize($this->thumb_url);
      }
      else if(!empty(SETTING_SEO_META_IMAGE_URL)) {
         $meta_image_url = SETTING_SEO_META_IMAGE_URL;
         list($meta_image_width, $meta_image_height) = getimagesize(str_replace(base_url(), '', SETTING_SEO_META_IMAGE_URL));
      }
      $meta_data['meta_image_url'] = $meta_image_url;
      $meta_data['meta_image_width'] = !empty($meta_image_width) ? $meta_image_width:'';
      $meta_data['meta_image_height'] = !empty($meta_image_height) ? $meta_image_height:'';


      return $meta_data;
   }
}